from typing import Optional
from .dict import DotDict


class Detector:
    def __init__(
        self,
        name: str = "01",
        is_effective: bool = True,
    ):
        self.name: str = name
        self.is_effective: bool = is_effective

    def __repr__(self):
        return f"<Detector: {self.name}, is_effective: {self.is_effective}>"


# d = Detector(name="01", is_effective=True)


class SimpleInstrument(DotDict):
    def __init__(self, name: str, detectors: list[Detector]):
        self._name = name
        super().__init__(**{d.name: d for d in detectors})

    @property
    def name(self):
        return self._name

    @property
    def detectors(self):
        return list(self.values())

    @property
    def detector_names(self):
        return list(self.keys())

    @property
    def n_detector(self):
        return len(self.detectors)

    @property
    def effective_detectors(self):
        return [d for d in self.detectors if d.is_effective]

    @property
    def effective_detector_names(self):
        return [d.name for d in self.detectors if d.is_effective]

    @property
    def n_effective_detector(self):
        return len(self.effective_detectors)

    def __repr__(self):
        return (
            f"<SimpleInstrument: {self.name}, "
            f"{self.n_detector} detectors, "
            f"{self.n_effective_detector} effective>"
        )


# detectors = [Detector(name=f"{_}") for _ in range(10)]
# si = SimpleInstrument(
#     name="SomeInstrument",
#     detectors=detectors,
# )


class ComplexInstrument(DotDict):
    def __init__(
        self,
        name: str,
        instruments: list[SimpleInstrument],
    ):
        super().__init__(**{i.name: i for i in instruments})
        self._name = name

    @property
    def name(self):
        return self._name

    @property
    def instruments(self):
        return list(self.values())

    @property
    def n_instrument(self):
        return len(self.instruments)

    @property
    def n_detector(self):
        return sum([i.n_detector for i in self.instruments])

    @property
    def n_effective_detector(self):
        return sum([i.n_effective_detector for i in self.instruments])

    @property
    def detectors(self):
        return [d for i in self.instruments for d in i.detectors]

    @property
    def detector_names(self):
        return [d.name for i in self.instruments for d in i.detectors]

    @property
    def effective_detectors(self):
        return [d for i in self.instruments for d in i.effective_detectors]

    @property
    def effective_detector_names(self):
        return [d.name for i in self.instruments for d in i.effective_detectors]

    def __repr__(self):
        str_instruments = ",".join([_.__repr__() for _ in self.instruments])
        return f"[ComplexInstrument: {self.name}, {str_instruments}]"


class Telescope(DotDict):
    def __init__(
        self,
        name: str,
        instruments: list[SimpleInstrument | ComplexInstrument],
    ):
        super().__init__(**{i.name: i for i in instruments})
        self._name = name

    @property
    def name(self):
        return self._name

    @property
    def instruments(self):
        return list(self.values())

    @property
    def n_instrument(self):
        return len(self.instruments)

    # def plan_to_detector(self, plan_data):
    #     # convert to dict
    #     plan_dict = dict(plan_data)
    #     if plan_dict["instrument"] == "HSTDM":


mbi = SimpleInstrument(
    name="MBI",
    detectors=[
        Detector(name=_, is_effective=True)
        for _ in [
            "06",
            "07",
            "08",
            "09",
            "11",
            "12",
            "13",
            "14",
            "15",
            "16",
            "17",
            "18",
            "19",
            "20",
            "22",
            "23",
            "24",
            "25",
        ]
    ],
)


sls = SimpleInstrument(
    name="SLS",
    detectors=[
        Detector(name=_, is_effective=True)
        for _ in [
            "01",
            "02",
            "03",
            "04",
            "05",
            "10",
            "21",
            "26",
            "27",
            "28",
            "29",
            "30",
        ]
    ],
)

ir = SimpleInstrument(
    name="IR",
    detectors=[Detector(name=f"IR{_}{_}", is_effective=False) for _ in range(1, 9)],
)

mci = SimpleInstrument(
    name="MCI",
    detectors=[Detector(name=_, is_effective=True) for _ in ["C1", "C2", "C3"]],
)

ifs = SimpleInstrument(
    name="IFS",
    detectors=[Detector(name=_, is_effective=True) for _ in ["B", "R"]],
)

cpic = SimpleInstrument(
    name="CPIC",
    detectors=[
        Detector(name="VIS", is_effective=True),
        Detector(name="NIR", is_effective=False),
    ],
)

hstdm = SimpleInstrument(
    name="HSTDM",
    detectors=[Detector(name=_) for _ in ["SIS1", "SIS2"]],
)

msc = ComplexInstrument(
    name="MSC",
    instruments=[mbi, sls, ir],
)

csst = Telescope(
    name="CSST",
    instruments=[msc, mci, ifs, cpic, hstdm],
)

# csst
# csst.MSC
# csst.MSC.MBI
# csst.MSC.SLS
# csst.MSC.IR
# csst.MCI
# csst.IFS
# csst.CPIC
# csst.HSTDM
# csst.HSTDM.effective_detectors
# csst.HSTDM.n_effective_detector
