from .common import request, Result

def current() -> Result:
    """
    获取当前登录用户信息
    
    Returns:
        Result: 当前用户信息
    """
    return request.get(f"/api/users/me")
def find(key: str, page: int = 1, limit: int = 0) -> Result:
    """
    查询用户信息
    
    Args:
        key (str): 查询键，如'username'或'email'
        page (int, optional): 分页页码，默认值为1
        limit (int, optional): 分页每页数量，默认值为0
    
    Returns:
        Result: 查询结果
    """
    params = {
        'key': key,
        'page': page,
        'limit': limit
    }
    return request.post(f"/api/users/find", params)

def login(username: str, password: str) -> Result:
    """
    用户登录
    
    Args:
        username (str): 用户名
        password (str): 密码
    
    Returns:
        Result: 登录结果
    """
    params = {
        'username': username,
        'password': password
    }
    return request.post(f"/api/users/login", params)

def add(username: str, password: str, nickname: str, email: str, role: str = 'user') -> Result:
    """
    添加用户
    
    Args:
        username (str): 用户名
        password (str): 密码
        nickname (str, optional): 姓名
        email (str): 邮箱
        role (str, optional): 角色，默认值为'user'
    
    Returns:
        Result: 添加结果
    """
    params = {
        'username': username,
        'password': password,
        'email': email,
        'nickname': nickname,
        'role': role
    }
    return request.post(f"/api/users/add", params)

def update(username: str, nickname: str, email: str, role: str = 'user') -> Result:
    """
    更新用户信息
    
    Args:
        username (str): 用户名
        nickname (str, optional): 姓名
        email (str): 邮箱
        role (str, optional): 角色，默认值为'user'
    
    Returns:
        Result: 更新结果
    """
    params = {
        'username': username,
        'nickname': nickname,
        'email': email,
        'role': role
    }
    return request.post(f"/api/users/update", params)

def passwd(old_password: str, new_password: str) -> Result:
    """
    更新用户密码
    
    Args:
        old_password (str): 旧密码
        new_password (str): 新密码
    
    Returns:
        Result: 更新结果
    """
    params = {
        'old_password': old_password,
        'new_password': new_password
    }
    return request.post(f"/api/users/password", params)

def delete(username: str) -> Result:
    """
    删除用户
    
    Args:
        username (str): 用户名
    
    Returns:
        Result: 删除结果
    """
    params = {
        'username': username
    }
    return request.post(f"/api/users/delete", params)
