import requests
import json
from enum import Enum

class Result(dict):
    '''
    类Result用于封装返回结果，包含success、message、data三个属性:
        success (bool): 是否成功
        message (str): 消息
        data (any): 数据
    '''
    __setattr__ = dict.__setitem__
    __getattr__ = dict.__getitem__
    def __init__(self):
        super(Result, self).__init__()
        self["code"] = 200
        self["message"] = ""
        self["data"] = None
    
    @property
    def success(self) -> bool:
        return self["code"] == 200

    @property
    def data(self) -> any:
        return self["data"]

    @property
    def message(self) -> str:
        return str(self["message"])
        
    @staticmethod
    def error(code: int = 500, message: str = "") -> "Result":
        r = Result()
        r["code"] = code
        r["message"] = message
        return r

    @staticmethod
    def ok_data(data = None) -> "Result":
        r = Result()
        r["code"] = 200
        r["data"] = data
        return r
        
    @staticmethod
    def ok_msg(message: str) -> "Result":
        r = Result()
        r["message"] = message
        return r

    def append(self, k: str, v) -> "Result":
        self[k] = v
        return self
    
    @classmethod
    def from_response(cls, resp: requests.Response) -> "Result":
        self = cls()
        self["code"] = resp.status_code
        if self.success:
            content_dict = json.loads(resp.content.decode('utf-8'))
            if isinstance(content_dict, dict) and 'total_count' in content_dict and 'records' in content_dict:
                self["data"] = content_dict['records']
                self["total_count"] = content_dict['total_count']
            else:
                self["data"] = content_dict
            self['data'] = DictObject.from_json(self['data'])
            self['message'] = ''
        else:
            try:
                content_dict = json.loads(resp.content.decode('utf-8'))
                self["message"] = content_dict.get("detail", "")
            except Exception as _:
                self["message"] = resp.content.decode('utf-8')
        return self

class DictObject(dict):
    def __init__(self, **kwargs):
        super(DictObject, self).__init__(**kwargs)
    def __getattr__(self, name):
        if name in self.keys():
            return self[name]
        return None
    @staticmethod
    def from_json(data):
        if not data:
            return data
        if isinstance(data, list):
            return [DictObject.from_json(i) for i in data]
        elif isinstance(data, dict):
            return DictObject(**data)
        else:
            return data
    
class ModuleID(Enum):
    MSC = "MSC"
    IFS = "IFS"
    MCI = "MCI"
    HSTDM = "HSTDM"
    CPIC = "CPIC"