from .version import __version__
from .common import request
from io import BytesIO
from .exceptions.exception import AppError
__all__ = [
    "__version__",
    "download_file",
    "read_file",
]
def download_file(file_path: str) -> bytes:
    """
    下载文件。
    
    Args:
        file_path (str): 相对文件路径。
    
    Returns:
        bytes: 文件内容。
    
    Raises:
        Exception: 如果下载失败，则抛出异常AppError。
    """
    response = request.download_file(f"/api/common/download/file?file_path={file_path}")
    if not response.ok:
        raise AppError("Failed to download the file. Reason: " + response.reason)
    return response.content

def read_file(file_path: str) -> BytesIO:
    """
    读取文件。
    
    Args:
        file_path (str): 相对文件路径。
    
    Returns:
        BytesIO: 文件内容。
    
    Raises:
        Exception: 如果读取失败，则抛出异常AppError。
    """
    from .common import fs
    with fs.get_file_storage().read_file(file_path) as iobytes_io:
        try:
            iobytes_io.seek(0)
        except OSError:
            raise AppError("Failed to read file.")
        return iobytes_io
    
def get_free_header(file_path: str) -> dict:
    result = request.get(f"/api/common/header?file_path={file_path}")
    if result.success:
        return result.data
    raise AppError("Failed to get free header. Reason: " + result.message)