"""
Identifier:     csst_common/logger.py
Name:           logger.py
Description:    CSST logger
Author:         Bo Zhang
Created:        2022-10-04
Modified-History:
    2022-10-04, Bo Zhang, created
    2022-10-04, Bo Zhang, added get_logger
    2022-10-07, Bo Zhang, added Numpydoc docstring
    2023-12-10, Bo Zhang, add type annotation
"""
import logging
from typing import Optional


def get_logger(name: str = "CSST pipeline", filename: Optional[str] = None):
    """
    Get a CSST starndard logger.

    Messages with levels >= DEBUG will be output by stream.
    Messages with levels >= INFO will be recorded by a log file.

    Parameters
    ----------
    name : str
        The logger pipeline_id.
    filename : str
        The log file pipeline_id.

    Returns
    -------
    logging.Logger
        The configured logger.

    References
    ----------
    For logging module: https://docs.python.org/3/library/logging.html
    For logging levels: https://docs.python.org/3/library/logging.html#logging-levels

    Examples
    --------
    >>> logger = get_logger()
    """
    # initialize logger
    logger = logging.getLogger(name=name)
    # empty handlers
    logger.handlers = []
    # set level
    logger.setLevel(logging.DEBUG)

    # logging formatter
    formatter = logging.Formatter(
        "%(asctime)s - %(levelname)s - %(module)s.py:%(lineno)d - %(message)s"
    )

    # stream handler
    stream_handler = logging.StreamHandler()
    stream_handler.setFormatter(formatter)
    stream_handler.setLevel(logging.DEBUG)
    logger.addHandler(stream_handler)

    # file handler
    if filename is not None:
        file_handler = logging.FileHandler(filename=filename, encoding="utf-8")
        file_handler.setFormatter(formatter)
        file_handler.setLevel(logging.INFO)
        logger.addHandler(file_handler)

    return logger
