"""
Identifier:     csst_common/dfs.py
Name:           dfs.py
Description:    DFS wrapper
Author:         Bo Zhang
Created:        2023-07-08
Modified-History:
    2023-12-22, Bowei Zhao, implemented convert_slsconf
    2023-12-22, Bo Zhang, tweaks

Details:

chipid: [01,   02,   03,   04,   05,   10,   21,   26,   27,   28,   29,   30]
filter: [GI,   GV,   GU,   GU,   GV,   GI,   GI,   GV,   GU,   GU,   GV,   GI]
clabel: [GI-1, GV-1, GU-1, GU-2, GV-2, GI-2, GI-3, GV-3, GU-3, GU-4, GV-4, GI-4]

keys = ['BEAMA', 'MMAG_EXTRACT_A', 'MMAG_MARK_A', 'DYDX_ORDER_A', 'DYDX_A_0', 'DYDX_A_1', 'XOFF_A', 'YOFF_A',
        'DISP_ORDER_A', 'DLDP_A_0', 'DLDP_A_1', 'BEAMB', 'MMAG_EXTRACT_B', 'MMAG_MARK_B', 'DYDX_ORDER_B', 'DYDX_B_0',
        'XOFF_B', 'YOFF_B', 'DISP_ORDER_B', 'DLDP_B_0', 'DLDP_B_1', 'BEAMC', 'MMAG_EXTRACT_C', 'MMAG_MARK_C',
        'DYDX_ORDER_C', 'DYDX_C_0', 'DYDX_C_1', 'XOFF_C', 'YOFF_C', 'DISP_ORDER_C', 'DLDP_C_0', 'DLDP_C_1', 'BEAMD',
        'MMAG_EXTRACT_D', 'MMAG_MARK_D', 'DYDX_ORDER_D', 'DYDX_D_0', 'DYDX_D_1', 'XOFF_D', 'YOFF_D', 'DISP_ORDER_D',
        'DLDP_D_0', 'DLDP_D_1', 'BEAME', 'MMAG_EXTRACT_E', 'MMAG_MARK_E', 'DYDX_ORDER_E', 'DYDX_E_0', 'DYDX_E_1',
        'XOFF_E', 'YOFF_E', 'DISP_ORDER_E', 'DLDP_E_0', 'DLDP_E_1']
"""
import json
import os
from astropy.io import fits


def convert_slsconf(
    extract1d_json_path: str,
    sensitivity_fits_path: str,
    dir_output: str,
) -> dict:
    extract1d_name = os.path.basename(extract1d_json_path)
    sensitivity_name = os.path.basename(sensitivity_fits_path)
    slsconf = dict()
    # conf_names = []
    # conf_paths = []
    sensitivity_order_names = []
    # sensitivity_order_paths = []
    # save CSST_MSC_MS_SENSITIVITY_{CHIPID}_{VERSION}_{ORDER}.conf
    for order in ["0st", "+1st", "-1st", "+2st", "-2st"]:
        sensitivity_order_name = sensitivity_name.replace(".fits", f"_{order}.fits")
        sensitivity_order_path = os.path.join(dir_output, sensitivity_order_name)
        resave_sensitivity(sensitivity_fits_path, sensitivity_order_path)
        sensitivity_order_names.append(sensitivity_order_name)
        # sensitivity_order_paths.append(sensitivity_order_path)
        slsconf[f"sensitivity_{order}"] = sensitivity_order_path

    # save CSST_MSC_MS_EXTRACT1D_{CHIPID}_{VERSION}_{DIRECTION}.conf
    for direction, GRATINGLR in zip(["left", "right"], ["GRATINGL", "GRATINGR"]):
        conf_name = extract1d_name.replace(".json", f"_{direction}.conf")
        conf_path = os.path.join(dir_output, conf_name)
        fsave_conf(extract1d_json_path, conf_path, sensitivity_order_names, GRATINGLR)
        # conf_names.append(conf_name)
        # conf_paths.append(conf_path)
        slsconf[f"conf_{direction}"] = conf_path

    # return dict(zip(conf_names, conf_paths)), dict(
    #     zip(sensitivity_order_names, sensitivity_order_paths)
    # )
    return slsconf


def readjson(file_json_path):
    with open(file_json_path) as f:
        d = json.load(f)
    return d


def fwriteKEY(fsx, flt):
    fsx.write("INSTRUMENT CSSTSLS" + "\n")
    fsx.write("CAMERA " + flt + "\n")
    if flt == "GI":
        fsx.write("WAVELENGTH 6200 10000" + "\n")
    elif flt == "GV":
        fsx.write("WAVELENGTH 4000 6200" + "\n")
    elif flt == "GU":
        fsx.write("WAVELENGTH 2550 4000" + "\n")
    fsx.write("\n" + "SCIENCE_EXT SCI ; Science extension" + "\n")
    fsx.write("DQ_EXT DQ       ; DQ extension" + "\n")
    fsx.write("ERRORS_EXT ERR  ; Error extension" + "\n")
    fsx.write("FFNAME csstFlat.fits" + "\n")
    fsx.write("DQMASK 246    ; 4096 and 512 taken out" + "\n")
    fsx.write("\n" + "RDNOISE 5.0" + "\n")
    fsx.write("EXPTIME EXPTIME" + "\n")
    fsx.write("POBJSIZE 1.0" + "\n")
    fsx.write("#SMFACTOR 1.0" + "\n\n")


def fwriteBEAM(
    fsx,
    extract1d_json_path,
    sensitivity_order_name,
    GRATINGLR,
    BEAMX,
    SENSITIVITY_X,
    MMAG_EXTRACT_X,
    MMAG_MARK_X,
    DYDX_ORDER_X,
    DYDX_X_0,
    DYDX_X_1,
    XOFF_X,
    YOFF_X,
    DISP_ORDER_X,
    DLDP_X_0,
    DLDP_X_1,
):
    d = readjson(extract1d_json_path)
    fsx.write(BEAMX), [
        fsx.write(" " + str(d[GRATINGLR][BEAMX][j]))
        for j in range(len(d[GRATINGLR][BEAMX]))
    ], fsx.write("\n")

    fsx.write(MMAG_EXTRACT_X + " " + str(d[GRATINGLR][MMAG_EXTRACT_X]) + "\n")
    fsx.write(MMAG_MARK_X + " " + str(d[GRATINGLR][MMAG_MARK_X]) + "\n")
    fsx.write("# " + "\n")
    fsx.write("# Trace description " + "\n")
    fsx.write("# " + "\n")
    fsx.write(DYDX_ORDER_X + " " + str(d[GRATINGLR][DYDX_ORDER_X]) + "\n")
    fsx.write(DYDX_X_0), [
        fsx.write(" " + str(d[GRATINGLR][DYDX_X_0][j]))
        for j in range(len(d[GRATINGLR][DYDX_X_0]))
    ], fsx.write("\n")
    if BEAMX == "BEAMB":
        pass
    else:
        fsx.write(DYDX_X_1), [
            fsx.write(" " + str(d[GRATINGLR][DYDX_X_1][j]))
            for j in range(len(d[GRATINGLR][DYDX_X_1]))
        ], fsx.write("\n")
    fsx.write("# " + "\n")
    fsx.write("# X and Y Offsets " + "\n")
    fsx.write("# " + "\n")
    fsx.write(XOFF_X + " " + str(d[GRATINGLR][XOFF_X]) + "\n")
    fsx.write(YOFF_X + " " + str(d[GRATINGLR][YOFF_X]) + "\n")
    fsx.write("# " + "\n")
    fsx.write("# Dispersion solution " + "\n")
    fsx.write("# " + "\n")
    fsx.write(DISP_ORDER_X + " " + str(d[GRATINGLR][DISP_ORDER_X]) + "\n")
    fsx.write(DLDP_X_0), [
        fsx.write(" " + str(d[GRATINGLR][DLDP_X_0][j]))
        for j in range(len(d[GRATINGLR][DLDP_X_0]))
    ], fsx.write("\n")
    fsx.write(DLDP_X_1), [
        fsx.write(" " + str(d[GRATINGLR][DLDP_X_1][j]))
        for j in range(len(d[GRATINGLR][DLDP_X_1]))
    ], fsx.write("\n")
    fsx.write("# " + "\n")
    fsx.write(SENSITIVITY_X + " " + sensitivity_order_name + "\n")
    fsx.write("# " + "\n" + "\n")


def fsave_conf(
    extract1d_json_path, extract1d_conf_path, sensitivity_order_names, GRATINGLR
):
    c = extract1d_conf_path
    flt = readjson(extract1d_json_path)["FILTER"]

    os.system("> " + c)
    fs = open(c, "a")
    fwriteKEY(fs, flt)
    fs.write("# 1 order (BEAM A) *******************" + "\n")
    fwriteBEAM(
        fs,
        extract1d_json_path,
        sensitivity_order_names[0],
        GRATINGLR,
        "BEAMA",
        "SENSITIVITY_A",
        "MMAG_EXTRACT_A",
        "MMAG_MARK_A",
        "DYDX_ORDER_A",
        "DYDX_A_0",
        "DYDX_A_1",
        "XOFF_A",
        "YOFF_A",
        "DISP_ORDER_A",
        "DLDP_A_0",
        "DLDP_A_1",
    )

    fs.write("\n# 0 order (BEAM B) *******************" + "\n")
    fwriteBEAM(
        fs,
        extract1d_json_path,
        sensitivity_order_names[1],
        GRATINGLR,
        "BEAMB",
        "SENSITIVITY_B",
        "MMAG_EXTRACT_B",
        "MMAG_MARK_B",
        "DYDX_ORDER_B",
        "DYDX_B_0",
        "DYDX_B_1",
        "XOFF_B",
        "YOFF_B",
        "DISP_ORDER_B",
        "DLDP_B_0",
        "DLDP_B_1",
    )
    fs.write("\n# -1 order (BEAM C) *******************" + "\n")
    fwriteBEAM(
        fs,
        extract1d_json_path,
        sensitivity_order_names[2],
        GRATINGLR,
        "BEAMC",
        "SENSITIVITY_C",
        "MMAG_EXTRACT_C",
        "MMAG_MARK_C",
        "DYDX_ORDER_C",
        "DYDX_C_0",
        "DYDX_C_1",
        "XOFF_C",
        "YOFF_C",
        "DISP_ORDER_C",
        "DLDP_C_0",
        "DLDP_C_1",
    )
    fs.write("\n# 2 order (BEAM D) *******************" + "\n")
    fwriteBEAM(
        fs,
        extract1d_json_path,
        sensitivity_order_names[3],
        GRATINGLR,
        "BEAMD",
        "SENSITIVITY_D",
        "MMAG_EXTRACT_D",
        "MMAG_MARK_D",
        "DYDX_ORDER_D",
        "DYDX_D_0",
        "DYDX_D_1",
        "XOFF_D",
        "YOFF_D",
        "DISP_ORDER_D",
        "DLDP_D_0",
        "DLDP_D_1",
    )
    fs.write("\n# -2 order (BEAM E) *******************" + "\n")
    fwriteBEAM(
        fs,
        extract1d_json_path,
        sensitivity_order_names[4],
        GRATINGLR,
        "BEAME",
        "SENSITIVITY_E",
        "MMAG_EXTRACT_E",
        "MMAG_MARK_E",
        "DYDX_ORDER_E",
        "DYDX_E_0",
        "DYDX_E_1",
        "XOFF_E",
        "YOFF_E",
        "DISP_ORDER_E",
        "DLDP_E_0",
        "DLDP_E_1",
    )
    fs.close()


def resave_sensitivity(sensitivity_fits_path, sensitivity_order_path):
    h = fits.open(sensitivity_fits_path)
    for extname in ["L0st", "LP1st", "LM1st", "LP2st", "LM2st"]:
        hdu0 = fits.PrimaryHDU()
        hdu1 = fits.BinTableHDU(h[extname].data)
        hdul = fits.HDUList([hdu0, hdu1])
        hdul.writeto(sensitivity_order_path, overwrite=True, checksum=True)
        hdul.close()

    h.close()
