"""
Identifier:     csst_common/io.py
Name:           io.py
Description:    IO module
Author:         Bo Zhang
Created:        2023-12-13
Modified-History:
    2023-12-10, Bo Zhang, created
    2023-12-15, Bo Zhang, added verify_checksum and append_header, add module header
    2023-12-16, Bo Zhang, fixed a bug in finding the first key when reformatting
    2024-06-08, Bo Zhang, added delete_section
    2025-08-05, Bo Zhang, added generate_meta
"""

import bisect
import json
import os
import uuid
import warnings
from copy import deepcopy
from typing import Optional

from astropy.io import fits
from astropy import table

from .time import now
from .utils import table_to_hdu, hdu_to_table


# meta字段和默认值
REQUIRED_KEYS_IN_META = dict(
    # 编排信息
    dataset=None,
    instrument=None,
    obs_type=None,
    obs_group=None,
    obs_id=None,
    # 探测信息
    detector=None,
    filter=None,
    # 参考信息
    pmapname=None,
    ref_cat=None,
    # 数据处理信息
    custom_id=None,
    batch_id=None,
    dag_group=None,
    dag_group_run=None,
    dag=None,
    dag_run=None,
    priority=-1,
    data_list=[],
    extra_kwargs={},
    created_time="1970-01-01T00:00:00.000",
    rerun=-1,
    # 数据产品信息
    data_model=None,  # 数据产品类型，手动设置
    data_uuid=None,  # UUID，自动设置
    qc_status=-1024,  # QC状态
    # Docker镜像名称和版本
    docker_image=None,  # 镜像名称，自动设置
    build=None,  # 镜像版本，自动设置
    # 额外的观测筛选参数
    object=None,  # 观测目标
    proposal_id=None,  # 观测申请ID
    ra=-3.141592653589793,  # 赤经
    dec=-3.141592653589793,  # 赤纬
    healpix=-1,  # HEALPix，每种数据产品的nside可以不一样
    obs_date="1970-01-01T00:00:00.000",  # 观测时间
    prc_date="1970-01-01T00:00:00.000",  # 处理时间
)


def generate_meta(**kwargs) -> dict:
    """
    Generate metadata.

    Parameters
    ----------
    kwargs : Any
        Metadata.

    Returns
    -------
    fits.Header
        Meta data header.
    """
    # copy REQUIRED_KEYS_IN_META
    meta = deepcopy(REQUIRED_KEYS_IN_META)
    # update meta with kwargs
    for k in kwargs.keys():
        if k in meta.keys():
            # 类型必须兼容
            # assert isinstance(
            #     kwargs[k], type(meta[k])
            # ), f"类型不兼容: kwargs['{k}'] 应为 {type(meta[k]).__name__} 或其子类"
            # # 赋值
            meta[k] = kwargs[k]
        else:
            raise KeyError(f"未知的meta参数: {k}")
    # automatically set docker_image, build, and created_date
    meta["docker_image"] = os.getenv("PIPELINE_ID", default=None)
    meta["build"] = os.getenv("BUILD", default=None)
    meta["prc_date"] = now()
    meta["data_uuid"] = str(uuid.uuid4())
    return meta


def append_meta(hdulist: fits.HDUList, meta: dict) -> fits.HDUList:
    """
    Append meta to hdulist.

    Parameters
    ----------
    hdulist : fits.HDUList
        HDUList.
    meta : dict
        Meta data dict.

    Returns
    -------
    fits.HDUList
        HDUList with meta.

    """
    # convert meta to fits header
    meta_card = fits.Card(
        keyword="META",
        value=json.dumps(meta, separators=(",", ":"), ensure_ascii=False),
    )
    meta_header = reformat_header(fits.Header(cards=[meta_card]), comment="METADATA")
    # append meta_header to hdulist[0].header
    hdulist[0].header = append_header(hdulist[0].header, meta_header)
    return hdulist


def extract_meta(hdulist: fits.HDUList) -> dict:
    """
    Extract meta from hdulist.

    Parameters
    ----------
    hdulist : fits.HDUList
        HDUList.

    Returns
    -------
    dict
        Meta data.
    """
    meta = json.loads(hdulist[0].header["META"])
    return meta


def verify_checksum(file_path) -> bool:
    """
    Verify a .fits file via checksum.

    Return True if checksum is good.

    Parameters
    ----------
    file_path : str
        File path.

    References
    ----------
    https://docs.astropy.org/en/stable/io/fits/usage/verification.html#verification-using-the-fits-checksum-keyword-convention
    """
    with warnings.catch_warnings(record=True) as warning_list:
        # file_path = fits.util.get_testdata_filepath('checksum_false.fits')
        with fits.open(file_path, checksum=True):
            pass
        print(warning_list)
    return len(warning_list) == 0


def append_header(
    h1: fits.Header,
    h2: fits.Header,
    duplicates: str = "delete",
) -> fits.Header:
    """
    Append h2 to h1.

    Append fits headers, taken into account duplicated keywords.

    Parameters
    ----------
    h1 : fits.Header
        Original fits header.
    h2 : fits.Header
        Extended fits header.
    duplicates : str
        The operation for processing duplicates.
        "delete" to delete duplicated keywords in h1 and keep them in h2.
        "update" to update duplicated keywords in h1 and remove them from h2.

    Returns
    -------
    fits.Header
        The combined header.

    References
    ----------
    https://docs.astropy.org/en/stable/io/fits/usage/headers.html#comment-history-and-blank-keywords
    """
    # copy data
    original_h = deepcopy(h1)
    extended_h = deepcopy(h2)
    original_keys = tuple(original_h.keys())
    extended_keys = tuple(extended_h.keys())
    ignored_keys = ("", "COMMENT", "HISTORY")
    assert duplicates in ("delete", "update")
    if duplicates == "update":
        for k in extended_keys:
            # print(card.keyword)
            if k not in ignored_keys and k in original_keys:
                print(f"Update existing key *{k} in original fits.Header*")
                original_h.set(k, extended_h[k], extended_h.comments[k])
                extended_h.remove(k)
    elif duplicates == "delete":
        for k in extended_keys:
            # print(card.keyword)
            if k not in ignored_keys and k in original_keys:
                print(f"Delete existing key *{k} in original fits.Header*")
                original_h.remove(k)
    original_h.extend(extended_h, bottom=True)
    return original_h


def reformat_header(
    h: fits.Header,
    strip: bool = True,
    comment: Optional[str] = None,
):
    """
    Reformat fits Header.

    Remove blanks, comments, and history, and add a new comment at the beginning.

    Parameters
    ----------
    h : fits.Header
        The original header.
    strip: bool = True
        If `True`, strip cards like SIMPLE, BITPIX, etc.
        so the rest of the header can be used to reconstruct another kind of header.
    comment: Optional[str] = None
        The new comment at the beginning.

    See Also
    --------
    astropy.io.fits.Header.strip()

    References
    ----------
    https://docs.astropy.org/en/stable/io/fits/api/headers.html#astropy.io.fits.Header.strip
    """
    h_copy = deepcopy(h)
    # strip
    if strip:
        h_copy.strip()
    # remove blanks, comments, and history
    h_copy.remove("", ignore_missing=True, remove_all=True)
    h_copy.remove("COMMENT", ignore_missing=True, remove_all=True)
    h_copy.remove("HISTORY", ignore_missing=True, remove_all=True)
    # add new comment
    assert len(h.cards) > 0
    first_key = list(h_copy.keys())[0]
    h_copy.add_comment("=" * 72, before=first_key)
    h_copy.add_comment(comment, before=first_key)
    h_copy.add_comment("=" * 72, before=first_key)
    return h_copy


def delete_section(
    header: fits.Header,
    title="WORLD COORDINATE SYSTEM INFORMATION",
) -> fits.Header:
    """
    Delete a section from a fits header.

    This function is used to delete the section of WCS information from the header.

    Parameters
    ----------
    header: fits.Header
        Header.
    title: str
        Title of the section to be deleted.

    Returns
    -------
    fits.Header
        Edited header.
    """

    # find sep lines
    i_seps = []
    for i_card, card in enumerate(header.cards):
        if (
            card.keyword == "COMMENT"
            and isinstance(card.value, str)
            and card.value.startswith("===")
        ) or i_card == len(header.cards) - 1:
            i_seps.append(i_card)

    # find section title
    i_section = -1
    for i_card, card in enumerate(header.cards):
        if card.keyword == "COMMENT" and card.value == title:
            i_section = i_card
            break
    if i_section < 0:
        raise ValueError(f"Title not found: `{title}`")

    # find section start and stop
    idx = bisect.bisect(i_seps, i_section)
    i_section_start = i_seps[idx - 1]
    i_section_stop = i_seps[idx + 1]

    # delete keys in reverse order
    print(f"Delete keys {i_section_start} to {i_section_stop} ...")
    for i in range(i_section_stop - 1, i_section_start - 1, -1):
        del header[i]

    return header


def append_meta_to_table_hdu(hdu: fits.BinTableHDU, meta: dict) -> fits.BinTableHDU:
    """Append meta information to a table HDU.

    Parameters
    ----------
    hdu : fits.BinTableHDU
        Table HDU.
    meta : dict
        Meta information.
    """
    # extract table from HDU
    t = hdu_to_table(hdu)
    # determine number of rows
    n_rows = len(t)
    # extract subset of meta information
    sub_meta = {
        "healpix": meta["healpix"],
        "data_uuid": meta["data_uuid"],
    }
    # create meta table
    t_meta = table.Table([sub_meta] * n_rows, dtype=("i4", "U36"))
    # add meta columns to table
    t.add_columns(t_meta.columns)
    # convert table to HDU
    hdu = table_to_hdu(t)
    return hdu
