"""
Identifier:     csst_common/utils/_module_docstr.py
Name:           _module_docstr.py
Description:    module docstring
Author:         Bo Zhang
Created:        2022-09-06
Modified-History:
    2022-09-06, Bo Zhang, Added header_str
    2022-09-19, Bo Zhang, Added ModuleHeader class
    2023-12-15, Bo Zhang, reformat module headers
"""
import datetime
import glob
import os
import shutil

import numpy as np

PACKAGE_PATH = os.path.dirname(os.path.dirname(__file__))


with open(PACKAGE_PATH + "/data/module_header.txt", "r") as f:
    MODULE_DOCSTR = "".join(f.readlines())


def cmd_exists(cmd, verbose=False):
    """ check command existence """
    existence = shutil.which(cmd) is not None
    if verbose:
        print("check existence of command *{}* ... {}".format(cmd, existence))
    return existence


def cmds_exist(cmds, verbose=True):
    """ check existence of """
    existences = [cmd_exists(cmd, verbose=verbose) for cmd in cmds]
    assert all(existences)
    return


class ModuleHeader:
    """ initialize headers for *.py files in rootdir """
    @staticmethod
    def init_headers(rootdir="", identifier="SJ4", author="author", description="", ignore_init=False, write=True):
        """ initialize headers for *.py files in rootdir

        Parameters
        ----------
        rootdir: str
            root directory
        identifier: str
            the identifier, e.g., SJ4
        author: str
            author pipeline_id
        description: str
            description of the module
        ignore_init: bool
            if True, ignore __init__.py files
        write: bool
            if True, write headers to modules
            if False, print headers

        Examples
        --------
        If you want template headers directly written into files:
        >>> ModuleHeader.init_headers(
        >>>     "/Users/cham/CsstProjects/csst_common", identifier="SJ4", author="Bo Zhang", write=True)
        Otherwise:
        >>> ModuleHeader.init_headers(
        >>>     "/Users/cham/CsstProjects/csst_common", identifier="SJ4", author="Bo Zhang", write=False)
        """
        fps = glob.glob(rootdir + "/**/*.py")
        for fp in fps:
            if os.path.basename(fp) == "__init__.py" and ignore_init:
                pass
            else:
                # for other files:
                header_str = MODULE_DOCSTR.format(
                    identifier=identifier,
                    file_path=fp.replace(rootdir + "/", ""),
                    file_name=os.path.basename(fp),
                    description=description,
                    author=author,
                    date=get_present_date(),
                )

                if write:
                    # write header
                    with open(fp, "r") as f:
                        lines = f.readlines()
                    if lines[0] == "\"\"\"\n":
                        print(fp, " --> header exists")
                        continue
                    with open(fp, "w") as f:
                        print(fp, " --> writing header")
                        f.write("\"\"\"\n")
                        f.write(header_str)
                        f.write("\"\"\"\n")
                        f.writelines(lines)
                else:
                    # print header
                    print(fp)
                    print(header_str)


def get_present_date():
    """ get present date, e.g., 2022-09-06 """
    now = datetime.datetime.now()
    return now.strftime("%Y-%m-%d")


def insert_header(header_str, file_path):
    """ insert header_str to a file """
    with open(file_path, "r") as f:
        content = f.readlines()
    if content[0] == "\"\"\"\n":
        # header exists
        print("Found header in {} ... skipped".format(file_path))
    else:
        # header does not exist, insert one
        print("Insert header to {} ...".format(file_path))
        with open(file_path, "w") as f:
            f.write(header_str)
            f.writelines(content)
    return


def header_exists(file_path):
    pass


def delete_header(file_path):
    """ try to delete header for a file """
    with open(file_path, "r") as f:
        content = f.readlines()
    is_str_quote = np.where([line == "\"\"\"\n" for line in content])[0]
    if len(is_str_quote) == 2:
        header_start, header_stop = is_str_quote
        with open(file_path, "w") as f:
            f.writelines(content[header_stop+1])
        print("deleted header for {}".format(file_path))
    else:
        # no header found
        print("no header found in {}".format(file_path))
    return
