"""
Identifier:     csst_common/__init__.py
Name:           __init__.py
Description:    csst_common package
Author:         Bo Zhang
Created:        2022-09-13
Modified-History:
    2023-07-08, Bo Zhang, created
    2023-12-15, Bo Zhang, add module header
"""

import os

from ccds import client
from .slsconf import convert_slsconf


class CCDS:
    def __init__(
        self,
        ccds_root: str = "/ccds_root",
        ccds_cache: str = "/pipeline/ccds_cache",
    ):
        print("[CCDS] Setting CCDS root path ... ", end="")
        self.ccds_root = ccds_root
        print(self.ccds_root)

        print("[CCDS] Setting CCDS cache path ... ", end="")
        self.ccds_cache = ccds_cache
        print(self.ccds_cache)

        print("[CCDS] Setting CCDS environment variables ... ", end="")
        os.environ["CCDS_MODE"] = "remote"
        os.environ["CCDS_PATH"] = ccds_cache
        os.environ["CCDS_OBSERVATORY"] = "csst"
        print("Done")

        print("[CCDS] Query for observatory ... ", end="")
        self.observatory = client.get_default_observatory()
        print(self.observatory)
        assert (
            self.observatory == "csst"
        ), f"observatory [{self.observatory}] is not `csst`!"

        self.pmapname = self.operational_context = client.get_default_context(
            self.observatory
        )
        print(f"[CCDS] Query for operational_context = {self.operational_context}... ")

        self.is_available = False

    def __repr__(self):
        return (
            f"[CCDS] url='{os.getenv('CCDS_SERVER_URL', default='')}' >\n"
            # f" - is_available={self.is_available}\n"
            f" - observatory='{self.observatory}'\n"
            f" - operational_context='{self.operational_context}'\n"
            f" - ccds_root='{self.ccds_root}'\n"
            f" - ccdscache='{self.ccds_cache}'\n"
        )

    def get_refs(
        self,
        file_path: str = "./test.fits",
        pmapname: str = "",
    ):
        # Set pmapname
        if pmapname:
            print(f"[CCDS] Setting pmapname = {pmapname}... ")
            self.pmapname = pmapname
        try:
            # dump mappings
            client.dump_mappings(self.pmapname)
            self.is_available = True
        except BaseException as e:
            print(f"[CCDS] Failed to get pmapname = {self.pmapname}:", e.args)
            self.is_available = False

        # print(f"[CCDS] operational_context = {self.operational_context}")
        print(f"[CCDS] pmapname = {self.pmapname}")
        # print("get min header")
        hdrs = client.api.get_minimum_header(
            self.pmapname, file_path, ignore_cache=False
        )
        # {'CAMERA': 'MS',
        #  'CHIPID': '10',
        #  'DATE-OBS': '2028-09-16T07:20:59',
        #  'FILTER': 'GI',
        #  'INSTRUME': 'MSC',
        #  'REFTYPE': 'UNDEFINED'}
        # print("get best ref")
        refs = client.api.get_best_references(self.pmapname, hdrs)
        # {'bias': 'csst_msc_ms_bias_10_000001.fits',
        #  'dark': 'csst_msc_ms_dark_10_000001.fits',
        #  'ledflat': 'csst_msc_ms_ledflat_10_000001.fits',
        #  'shutter': "NOT FOUND  parameter='CHIPID' value='10' is not in ['06', '07', '08', '09', '11', '12', '13',
        #  '14', '15', '16', '17', '18', '19', '20', '22', '23', '24', '25']"}
        # assert ref_type in refs.keys(), f"ref_type [{ref_type}] not in {refs.keys()}"
        # print("get file info")
        refs_fp = dict()
        for ref_type in refs.keys():
            d = client.api.get_file_info(self.pmapname, refs[ref_type])
            if d:
                fp = os.path.join(self.ccds_root, d["file_path"])
                assert os.path.exists(fp), f"File path [{fp}] does not exist!"
                refs_fp[ref_type] = fp
            else:
                print(f"Failed to get file info for [{ref_type}:{refs[ref_type]}]")
        # {'bias': '/ccds_root/references/msc/csst_msc_ms_bias_10_000001.fits',
        #  'dark': '/ccds_root/references/msc/csst_msc_ms_dark_10_000001.fits',
        #  'ledflat': '/ccds_root/references/msc/csst_msc_ms_ledflat_10_000001.fits'}
        return refs_fp

    @staticmethod
    def convert_slsconf(refs: dict, dir_output: str) -> dict:
        return convert_slsconf(
            extract1d_json_path=refs["extract1d"],
            sensitivity_fits_path=refs["sensitivity"],
            dir_output=dir_output,
        )


"""
file_path="/dfsroot/L0/MSC/SCIE/62030/10160000105/MS/CSST_MSC_MS_SCIE_20280916072059_20280916072329_10160000105_10_L0_V01.fits"
hdrs = client.api.get_minimum_header(c.operational_context, file_path, ignore_cache=True)
refs = client.api.get_best_references(c.operational_context, hdrs)
refs = crds.getrecommendations(hdrs, reftypes=None, context=c.operational_context, ignore_cache=True, observatory=c.observatory, fast=False)
refs = crds.getreferences(hdrs, reftypes=['bias'], context=c.operational_context, ignore_cache=False, observatory=c.observatory)


docker run --rm -it \
    -v /share/crdsdata/data:/ccds_root \
    -e CRDS_SERVER_URL=http://172.24.27.2:29000 \
    -v /share/dfs:/dfsroot \
    csst/csst-msc-l1-mbi bash
"""

if __name__ == "__main__":
    c = CCDS()
    refs = c.get_refs()
    print(refs)

    refs = c.retry(
        c.get_refs,
        3,
        file_path="/dfsroot/L0/MSC/SCIE/62030/10160000105/MS/CSST_MSC_MS_SCIE_20280916072059_20280916072329_10160000105_11_L0_V01.fits",
    )
    print(refs)
    # {'bias': '/ccds_root/references/msc/csst_msc_ms_bias_11_000001.fits',
    #  'dark': '/ccds_root/references/msc/csst_msc_ms_dark_11_000001.fits',
    #  'ledflat': '/ccds_root/references/msc/csst_msc_ms_ledflat_11_000001.fits',
    #  'shutter': '/ccds_root/references/msc/csst_msc_ms_shutter_11_000001.fits'}
