"""
Identifier:     csst_common/status.py
Name:           status.py
Description:    CSST status
Author:         Bo Zhang
Created:        2022-10-04
Modified-History:
    2022-10-04, Bo Zhang, created
    2022-10-04, Bo Zhang, added CsstStatus
    2022-10-07, Bo Zhang, added Numpydoc docstring
    2023-12-19, Bo Zhang, updated example
"""
from enum import IntEnum
from typing import Optional, Any


class CsstStatus(IntEnum):
    """
    The CSST standard status class.

    This class provides a set of status which should be returned from each CSST algorithm module.
    0, 1, 2 stands for PERFECT, WARNING, ERROR, respectively.

    Examples
    --------
    >>> CsstStatus(0) # SUCCESS
    >>> CsstStatus(1) # WARNING
    >>> CsstStatus(2) # ERROR
    """

    # status list to be completed
    PERFECT = 0
    WARNING = 1
    ERROR = 2


class CsstResult:
    """
    CSST standard result class.

    This is designed for all algorithm modules.

    Parameters
    ----------
    status : CsstStatus
        The final status of the algorithm.
    files : Optional[list[str]]
        The files to which changes are made.
    **output : Any
        Additional results.

    Examples
    --------
    >>> CsstResult(CsstStatus.PERFECT, ["file1.fits", "file2.fits"])
    """

    def __init__(
        self,
        status: CsstStatus = CsstStatus.ERROR,
        files: Optional[list[str]] = None,
        **output: Any,
    ):
        if files is None:
            files = []
        assert isinstance(
            status, CsstStatus
        ), "`status` should be a `csst_common.CsstStatus` type objects."
        assert isinstance(
            files, list
        ), "`files` should be a list of `str` type objects."
        for _ in files:
            assert isinstance(_, str), "`files` should be a list of `str` type objects."
        self.status: CsstStatus = status
        self.files: Optional[list] = files
        self.output: dict = output

    def __repr__(self):
        return f""" <CsstResult status={self.status}>
 - files: {self.files}
 - output: {self.output} \n
"""
