"""
Identifier:     csst_common/__init__.py
Name:           __init__.py
Description:    csst_common package
Author:         Bo Zhang
Created:        2022-09-13
Modified-History:
    2023-07-08, Bo Zhang, created
    2023-12-15, Bo Zhang, add module header
"""

import os
import logging
from typing import Optional

from ccds import client


class CCDS:

    def __init__(
        self,
        use_oss=False,
        ccds_root: str = "/ccds_root",
        ccds_cache: str = "/pipeline/ccds_cache",
        logger: Optional[logging.Logger] = None,
    ):
        self.use_oss = use_oss
        self.ccds_root = ccds_root
        self.ccds_cache = ccds_cache
        self.logger = logger

        os.environ["CCDS_MODE"] = "remote"
        os.environ["CCDS_PATH"] = ccds_cache
        os.environ["CCDS_OBSERVATORY"] = "csst"

        self.observatory = client.get_default_observatory()
        self.operational_context = client.get_default_context(self.observatory)

    def get_refs(
        self,
        file_path: str,
        pmapname: Optional[str] = None,
    ) -> dict:

        if not pmapname:
            pmapname = self.operational_context

        # dump pmap
        client.dump_mappings(pmapname)

        # get header
        hdrs: dict = client.api.get_minimum_header(
            pmapname, file_path, ignore_cache=False
        )
        # {'CAMERA': 'MS',
        #  'DATE-OBS': '2026-02-25T03:58:41.0',
        #  'DETECTOR': '01',
        #  'GAINLVL': '01',
        #  'INSTRUME': 'MSC',
        #  'LEDTEMP': '173.0',
        #  'REFTYPE': 'UNDEFINED',
        #  'ROSPEED': '10.0'}
        refs: dict = client.api.get_best_references(pmapname, hdrs)
        # {'bias': 'csst_msc_ms_bias_01_000002.fits',
        #  'countmbi': 'csst_msc_ms_countmbi_00_000001.pickle',
        #  'countsls': 'csst_msc_ms_countsls_00_000001.pickle',
        #  'cti': 'csst_msc_ms_cti_01_000001.fits',
        #  'dark': 'csst_msc_ms_dark_01_000001.fits',
        #  'deepcr': 'csst_msc_ms_deepcr_01_000001.pickle',
        #  'detection': 'csst_msc_ms_detection_00_000001.pickle',
        #  'element': 'csst_msc_ms_element_00_000005.fits',
        #  'extract1d': 'csst_msc_ms_extract1d_01_000002.toml',
        #  'flat': "NOT FOUND  parameter='DETECTOR' value='01' is not in ['06', '07', '08', '09', '11', '12', '13', '14', '15', '16', '17', '18', '19', '20', '22', '23', '24', '25']",
        #  'gain': 'csst_msc_ms_gain_01_000003.fits',
        #  'ledflat': 'csst_msc_ms_ledflat_01_000002.fits',
        #  'ledflux': 'csst_msc_ms_ledflux_00_000005.fits',
        #  'ooccfg': 'csst_msc_ms_ooccfg_00_000001.toml',
        #  'pflat': "NOT FOUND  parameter='DETECTOR' value='01' is not in ['06', '07', '08', '09', '11', '12', '13', '14', '15', '16', '17', '18', '19', '20', '22', '23', '24', '25']",
        #  'photflat': "NOT FOUND  parameter='DETECTOR' value='01' is not in ['06', '07', '08', '09', '11', '12', '13', '14', '15', '16', '17', '18', '19', '20', '22', '23', '24', '25']",
        #  'psf': "NOT FOUND  parameter='DETECTOR' value='01' is not in ['06', '07', '08', '09', '11', '12', '13', '14', '15', '16', '17', '18', '19', '20', '22', '23', '24', '25']",
        #  'sensitivity': 'csst_msc_ms_sensitivity_01_000002.fits',
        #  'shutter': "NOT FOUND  parameter='DETECTOR' value='01' is not in ['06', '07', '08', '09', '11', '12', '13', '14', '15', '16', '17', '18', '19', '20', '22', '23', '24', '25']",
        #  'skyflat': "NOT FOUND  parameter='DETECTOR' value='01' is not in ['06', '07', '08', '09', '11', '12', '13', '14', '15', '16', '17', '18', '19', '20', '22', '23', '24', '25']",
        #  'zeropoint': 'csst_msc_ms_zeropoint_00_000010.fits'}
        if self.logger:
            self.logger.info(f"refs: {refs}")
        else:
            print(f"refs: {refs}")

        refs_abspath = {}

        if not self.use_oss:
            # kick process NOT FOUND cases
            for ref_type, ref_name in refs.items():
                ref_info = client.api.get_file_info(pmapname, ref_name)
                if ref_info:
                    ref_abspath = os.path.join(self.ccds_root, ref_info["file_path"])
                    assert os.path.exists(ref_abspath), ref_abspath
                    refs_abspath[ref_type] = ref_abspath
        else:
            # kick process NOT FOUND cases
            for ref_type, ref_name in refs.items():
                ref_info = client.api.get_file_info(pmapname, ref_name)
                if ref_info:
                    ref_abspath = ref_info["file_path"]
                    # assert os.path.exists(ref_abspath), ref_abspath
                    refs_abspath[ref_type] = ref_abspath

        if self.logger:
            self.logger.info(f"refs_abspath: {refs_abspath}")
        else:
            print(f"refs_abspath: {refs_abspath}")

        return dict(
            file_path=file_path,
            pmapname=pmapname,
            refs=refs_abspath,
        )

    def validate(self, pmapname: str) -> bool:
        """Validate pmap."""
        return client.api.check_mapping_exists(pmapname)

    def list(self):
        """List all pmap."""
        return client.api.list_mappings(glob_pattern="*.pmap")


if __name__ == "__main__":
    ccds = CCDS()
    file_path = "/dfs_root/CSST_L0/test-msc-c9-25sqdeg-v3/MSC/20260225/MS/CSST_MSC_MS_WIDE_20260225035843_20260225040113_10100285453_01_L0_V01.fits"
    pmapname = "csst_000094.pmap"
    ccds.get_refs(file_path, pmapname=pmapname, use_oss=False)
