"""
Identifier:     csst_common/fits.py
Name:           fits.py
Description:    IO module
Author:         Bo Zhang
Created:        2025-10-23
Modified-History:
    2025-10-23, Bo Zhang, created, support FITS operations for s3
"""

import io
import os.path

from astropy.io import fits as astropy_fits
from astropy.io.fits import *
from csst_fs.s3_config import load_s3_options
from typing import Any

s3_options = load_s3_options()

s3_prefix = ("s3://", "s3a://")


class HDUList(astropy_fits.HDUList):
    def to_bytes(self) -> bytes:
        """Convert the HDUList to bytes."""

        # 使用 with 语句自动管理资源
        with io.BytesIO() as memory_buffer:
            self.writeto(memory_buffer)
            fits_bytes = memory_buffer.getvalue()
        return fits_bytes


def prepare_fits_files_for_ingestion(files: list[str]) -> list[dict]:
    """Prepare FITS files for ingestion."""
    ingestion_files = []
    for file in files:
        with open(file) as hl:
            fits_bytes = hl.to_bytes()
        ingestion_files.append(
            {
                "file_name": os.path.basename(file),
                "file_content": fits_bytes,
            }
        )
    return ingestion_files


def open(filename, *args, **kwargs) -> HDUList:
    if filename.startswith(s3_prefix):
        # read FITS file from s3
        return HDUList(
            astropy_fits.open(
                filename, *args, use_fsspec=True, fsspec_kwargs=s3_options, **kwargs
            )
        )
    else:
        # read FITS file from local
        return HDUList(astropy_fits.open(filename, **kwargs))


def getheader(filename, *args, **kwargs) -> astropy_fits.Header:
    if filename.startswith(s3_prefix):
        # read FITS file from s3
        return astropy_fits.getheader(
            filename, *args, use_fsspec=True, fsspec_kwargs=s3_options, **kwargs
        )
    else:
        # read FITS file from local
        return astropy_fits.getheader(filename, **kwargs)


def getval(filename, *args, **kwargs) -> Any:
    if filename.startswith(s3_prefix):
        # read FITS file from s3
        return astropy_fits.getval(
            filename, *args, use_fsspec=True, fsspec_kwargs=s3_options, **kwargs
        )
    else:
        # read FITS file from local
        return astropy_fits.getval(filename, **kwargs)


def getdata(filename, *args, **kwargs) -> Any:
    if filename.startswith(s3_prefix):
        # read FITS file from s3
        return astropy_fits.getdata(
            filename, *args, use_fsspec=True, fsspec_kwargs=s3_options, **kwargs
        )
    else:
        # read FITS file from local
        return astropy_fits.getdata(filename, **kwargs)
