import time
import requests
from typing import Dict, List, Any, TypedDict, Optional, Tuple
from ..s3_config import load_backend_settings
from .pipeline_result import PipelineResult
from .dag_run import DagGroupRun, DagRun
from ..common import utils

_DAG_RUN_BATCH_SIZE: int = 512

class PipelineBatch(TypedDict):
    dag_group: str
    dag: str
    batch_id: str

DateTimeTuple = Tuple[str, str]

def new_dag_group_run(dag_group_run: DagGroupRun, dag_run_list: Optional[List[DagRun]] = None) -> PipelineResult:
    """
    Trigger a pipeline run for the provided DAG group run.

    Retries up to 3 times on transient/network errors for each batch.
    If dag_run_list is provided and exceeds batch size, splits it into batches of 512 runs.

    Args:
        dag_group_run: DagGroupRun TypedDict with keys: dag_group, dag_group_run, batch_id, priority, created_time, queue_time
        dag_run_list: Optional list of DagRun TypedDicts. If None or empty, submits without runs.

    Returns:
        PipelineResult containing the run_id(s). If multiple batches are submitted, returns the last successful result.
        If any batch fails, returns the failed result immediately.

    Raises:
        RuntimeError: on permanent failure or invalid response
    """
    api_url = load_backend_settings()['backend_url']
    if not api_url:
        raise RuntimeError("CSST backend api url is not set")
    pipeline_run_endpoint = f"{api_url}/level2/pipeline/run"

    # If dag_run_list is None or empty, submit once with empty list
    if dag_run_list is None or len(dag_run_list) == 0:
        return _submit_pipeline_batch(pipeline_run_endpoint, dag_group_run, [])

    # Process dag_run_list in batches
    results = []
    for i in range(0, len(dag_run_list), _DAG_RUN_BATCH_SIZE):
        batch = dag_run_list[i:i + _DAG_RUN_BATCH_SIZE]
        result = _submit_pipeline_batch(pipeline_run_endpoint, dag_group_run, batch)
        results.append(result)

        if not result.success:
            # If any batch fails, return the failed result immediately
            return result

    # If all batches succeed, return the last result
    return results[-1]


def _submit_pipeline_batch(endpoint: str, dag_group_run: DagGroupRun, dag_run_batch: List[DagRun]) -> PipelineResult:
    """
    Internal helper to submit a single batch of pipeline runs with retry logic.

    Args:
        endpoint: API endpoint URL
        dag_group_run: DagGroupRun configuration
        dag_run_batch: List of DagRun items (can be empty)

    Returns:
        PipelineResult with the API response

    Raises:
        RuntimeError: on permanent failure after all retries
    """
    max_retries = 3
    backoff_base = 1
    headers = {"Content-Type": "application/json"}

    for attempt in range(1, max_retries + 1):
        try:
            resp = requests.post(
                endpoint,
                json={"dag_group_run": dag_group_run, "dag_run_list": dag_run_batch},
                headers=headers,
                timeout=30
            )
            resp.raise_for_status()

            try:
                data = resp.json()
            except ValueError as e:
                raise RuntimeError(f"Invalid JSON response from pipeline API: {e}")

            if not data.get("success", False):
                raise RuntimeError(f"Pipeline API returned unsuccessful response: {data.get('code')} {data.get('message')}")

            result = data.get("result")
            if not isinstance(result, dict):
                raise RuntimeError(f"Unexpected pipeline API result shape: {data}")

            run_id = result.get("run_id")
            if run_id == None:
                raise RuntimeError(f"Pipeline API did not return run_id: {data}")

            pipeline_result: PipelineResult = PipelineResult()
            pipeline_result["code"] = data.get("code")
            pipeline_result["message"] = data.get("message")
            pipeline_result["data"] = data.get("result")
            return pipeline_result

        except (requests.RequestException, RuntimeError) as exc:
            if attempt == max_retries:
                raise RuntimeError(f"Failed to run pipeline after {max_retries} attempts: {exc}") from exc

            wait = backoff_base * (2 ** (attempt - 1))
            time.sleep(wait)



def query_run_state(run_id: str) -> Dict[str, Any]:
    """
    Query the state of a pipeline run given an id.

    Args:
        run_id: Run id of the pipeline run

    Returns:
        Dictionary of the format:
        {
            "state": "submission_pending",
        }
    """
    if not run_id:
        raise ValueError("run_id must be provided")

    api_url = load_backend_settings()['backend_url']
    if not api_url:
        raise RuntimeError("CSST backend api url is not set")

    endpoint = f"{api_url}/level2/pipeline/run/{run_id}"

    try:
        response = requests.get(endpoint, timeout=30)
        response.raise_for_status()
        data = response.json()
    except requests.RequestException as e:
        raise RuntimeError(f"Failed to query run state: {e}")

    if not data.get("success"):
        raise RuntimeError(f"Unexpected API response: {data}")

    result = data.get("result")
    if not result:
        return {"state": "not_found"}

    state = result.get("status")
    return {"state": state if state else "unknown"}

"""
Adapted from csst-dfs-client.
"""
def find_group_run(dag_group: Optional[str] = None,
        batch_id: Optional[str] = None,
        queue_time: Optional[DateTimeTuple] = None,
        prc_status: Optional[int] = None,
        page: int = 1,
        limit: int = 0):
    """
    根据给定的参数搜索DAG组的记录
    
    Args:
        dag_group (Optional[str], optional): DAG处理组. Defaults to None.
        batch_id (Optional[str], optional): 批次号. Defaults to None.
        queue_time (Optional[DateTimeTuple], optional): 入队时间范围. Defaults to None.
        prc_status (Optional[int], optional): 处理状态. Defaults to None.
        page (int, optional): 页码. Defaults to 1.
        limit (int, optional): 每页数量. Defaults to 0，不限制.
    
    Returns:
        Result: 搜索结果对象.
    
    """

    params = {
        'dagGroup': dag_group,
        'batchId': batch_id,
        'prcStatus': prc_status,
        'queueTimeStart': None,
        'queueTimeEnd': None,
        'page': page,
        'pageSize': limit if limit > 0 else None ,
    }
    
    if queue_time is not None:
        params['queueTimeStart'], params['queueTimeEnd'] = queue_time
        if params['queueTimeStart'] and utils.is_valid_datetime_format(params['queueTimeStart']):
            pass
        if params['queueTimeEnd'] and utils.is_valid_datetime_format(params['queueTimeEnd']):
            pass 
    
    headers = {"Content-Type": "application/json"}
    api_url = load_backend_settings()['backend_url']
    if not api_url:
        raise RuntimeError("CSST backend api url is not set")
    endpoint = f"{api_url}/level2/pipeline/search/dag_group_run"

    resp = requests.post(
        endpoint,
        json=params,
        headers=headers,
        timeout=30
    )
    resp.raise_for_status()

    try:
        data = resp.json()
    except ValueError as e:
        raise RuntimeError(f"Invalid JSON response from dag group run search API: {e}")

    if not data.get("success", False):
        raise RuntimeError(f"Dag group run search API returned unsuccessful response: {data.get('code')} {data.get('message')}")

    result = data.get("result")
    if not isinstance(result, list):
        raise RuntimeError(f"Unexpected dag group run search API result shape: {data}")

    pipeline_result: PipelineResult = PipelineResult()
    pipeline_result["code"] = data.get("code")
    pipeline_result["message"] = data.get("message")
    pipeline_result["data"] = data.get("result")
    return pipeline_result


"""
Adapted from csst-dfs-api.
"""
def update_dag_run(dag_run: str, status_code: int, queue_time: Optional[str] = None, 
                            start_time: Optional[str] = None, end_time: Optional[str] = None):
    """
    更新DAG运行的处理状态
    
    Args:
        dag_run (str): DAG运行标识
        status_code (int): 状态码
        queue_time (Optional[str], optional): 入队时间. Defaults to None.
        start_time (Optional[str], optional): 开始时间. Defaults to None.
        end_time (Optional[str], optional): 结束时间. Defaults to None.
    
    Returns:
        Result: 操作结果
    """

    params = {
        'dagRun': dag_run,
        'statusCode': status_code,
        'queueTime': queue_time,
        'startTime': start_time,
        'endTime': start_time
    }

    headers = {"Content-Type": "application/json"}
    api_url = load_backend_settings()['backend_url']
    if not api_url:
        raise RuntimeError("CSST backend api url is not set")
    endpoint = f"{api_url}/level2/pipeline/update/dag_run"

    resp = requests.put(
        endpoint,
        json=params,
        headers=headers,
        timeout=30
    )
    resp.raise_for_status()

    try:
        data = resp.json()
    except ValueError as e:
        raise RuntimeError(f"Invalid JSON response from dag run update API: {e}")

    if not data.get("success", False):
        raise RuntimeError(f"Dag group run update API returned unsuccessful response: {data.get('code')} {data.get('message')}")

    result = data.get("result")
    if not isinstance(result, bool):
        raise RuntimeError(f"Unexpected dag run update API result shape: {data}")

    pipeline_result: PipelineResult = PipelineResult()
    pipeline_result["code"] = data.get("code")
    pipeline_result["message"] = data.get("message")
    pipeline_result["data"] = data.get("result")
    return pipeline_result

