import requests
import time
from typing import Dict, List, Any, TypedDict
from csst_fs import s3_fs
from ..s3_config import load_backend_settings

class IngestionFile(TypedDict):
    file_name: str
    file_content: bytes

def start_ingestion_task(files: List[IngestionFile]) -> Dict[str, Any]:
    """
    Submit a list of files for ingestion by:
    1. Requesting an OSS upload path from the ingestion API.
    2. Uploading the file to the returned OSS path.
    3. Reporting the finished upload back to the ingestion API.

    Args:
        [
            {
                file_name (str): The file name for storing the file after ingestion.
                file_content (bytes): The file's content
            },
            {
                ...
            }
        ]

    Returns:
        dict: A dict containing a task_id referring to the queued processing task as well as a field failed, listing the file names for which ingestion failed.
        Example:
        {
            "task_id": "5",
            "failed": List[str] List of file names for which ingestion failed.
        }

    Raises:
        RuntimeError: If committing failed after retries.
    """
    api_url = load_backend_settings()['backend_url']
    if not api_url:
        raise RuntimeError("CSST backend api url is not set")
    request_upload_endpoint = f"{api_url}/datasync/level2/upload"

    succeeded_files: List[IngestionFile] = []
    failed_files: List[IngestionFile] = []

    for file in files:
        file_info = {"file_name": file["file_name"], "bucket": None, "key": None}
        try:
            # Step 1: Request an OSS upload path
            payload = {"fileName": file["file_name"]}

            try:
                response = requests.post(request_upload_endpoint, json=payload, timeout=30)
                response.raise_for_status()
                data = response.json()
            except requests.RequestException as e:
                raise RuntimeError(f"Failed to request OSS upload path: {e}")

            if not data.get("success") or "result" not in data:
                raise RuntimeError(f"Unexpected response while requesting upload path: {data}")

            oss_upload_bucket = data["result"].get("bucket")
            oss_upload_key = data["result"].get("key")

            if not oss_upload_bucket or not oss_upload_key:
                raise RuntimeError(f"No OSS upload bucket/key returned from API: {data}")

            file_info["bucket"] = oss_upload_bucket
            file_info["key"] = oss_upload_key

            # Step 2: Upload file to OSS path
            max_retries = 3
            backoff_factor = 2

            for attempt in range(1, max_retries + 1):
                try:
                    with s3_fs.open(f"s3://{oss_upload_bucket}/{oss_upload_key}", "wb") as f:
                        f.write(file["file_content"])
                        succeeded_files.append(file_info)
                        break
                except e:
                    if attempt == max_retries:
                        raise RuntimeError(f"OSS upload failed after {max_retries} attempts: {e}")
                    else:
                        wait_time = backoff_factor ** (attempt - 1)
                        print(f"OSS upload attempt {attempt} failed, retrying in {wait_time}s...")
                        time.sleep(wait_time)
        except RuntimeError as e:
            failed_files.append(file_info)


    # Step 3: Report upload completion, get task id
    try:
        report_upload_endpoint = f"{api_url}/datasync/level2/queue/batch"
        report_payload = [{"bucket": file["bucket"], "key": file["key"], "fileName": file["file_name"]} for file in succeeded_files]
        report_payload += [{"bucket": file["bucket"], "key": file["key"], "fileName": file["file_name"]} for file in failed_files]
        report_response = requests.post(report_upload_endpoint, json=report_payload, timeout=30)
        report_response.raise_for_status()
        report_data = report_response.json()
    except requests.RequestException as e:
        raise RuntimeError(f"Failed to report completed upload: {e}")

    if report_data.get("success") and "result" in report_data and "task_id" in report_data["result"]:
        return {"task_id": report_data["result"]["task_id"], "failed": [file["file_name"] for file in failed_files]}
    else:
        raise RuntimeError(f"Unexpected response while reporting upload: {report_data}")


def query_task_state(task_id: str) -> Dict[str, Any]:
    """
    Query the processing state of a processing task given a L2 task id.

    Args:
        task_id: Task id of the L2 processing task

    Returns:
        Dictionary of the format:
        {
            "state": "submission_pending",
        }
    """
    if not task_id:
        raise ValueError("task_id must be provided")

    api_url = load_backend_settings()['backend_url']
    if not api_url:
        raise RuntimeError("CSST backend api url is not set")

    endpoint = f"{api_url}/datasync/level2/{task_id}"

    try:
        response = requests.get(endpoint, timeout=30)
        response.raise_for_status()
        data = response.json()
    except requests.RequestException as e:
        raise RuntimeError(f"Failed to query task state: {e}")

    if not data.get("success"):
        raise RuntimeError(f"Unexpected API response: {data}")

    result = data.get("result")
    if not result:
        return {"state": "not_found"}

    sync_state = result.get("syncState")
    return {"state": sync_state if sync_state else "unknown"}
