import requests
from typing import Dict, Any, List
from ..s3_config import load_backend_settings

def query_metadata(
    filter: Dict[str, Any],
    key: List[str],
    hdu: int = 0,
    offset: int = 0,
    limit: int = 10,
) -> List[Dict[str, Any]]:
    """
    Query for file info by metadata values. Results are paginated using offset & limit and ordered ascending by database insert timestamp.

    Args:
        filter: The filter dict described below.
        key: A list of string values, corresponding to metadata keys that should be included in the output.
        hdu: The hdu the filter & key arguments refer to. Default is 0. E.g. 0, 1.
        offset: Optional offset for paginated results (default 0).
        limit: Optional limit for paginated results (default 10).
    Returns:
        A List[Dict] of matching documents containing a file_path value and the keys set as 'key' parameter under 'metadata'.
        E.g. with key = ["dataset", "instrument", "obs_group", "obs_id"]
            then returns:
            [
                {
                    "file_path": "CSST_L0/MSC/SCI/60310/10100000000/MS/CSST_MSC_MS_SCIE_20290225043953_20290225044223_10100000000_03_L0_V01.fits",
                    "metadata": {
                        "dataset":"csst-msc-c11-1000sqdeg-wide-test-v2",
                        "instrument":"MSC",
                        "obs_group":"W1",
                        "obs_id":"10200000000"
                    },
                },
            ]
    """

    if not filter:
        raise ValueError("Filter cannot be empty")

    api_url = load_backend_settings()['backend_url']
    if not api_url:
        raise RuntimeError("CSST backend api url is not set")

    endpoint = f"{api_url}/datasync/metadata/query"

    payload = {
        "filter": filter,
        "key": key,
        "hdu": hdu,
        "offset": offset,
        "limit": limit,
    }

    try:
        response = requests.post(endpoint, json=payload, timeout=30)
        response.raise_for_status()
    except requests.RequestException as e:
        raise RuntimeError(f"Metadata query failed: {e}")

    data = response.json()

    if not data.get("success") or "result" not in data:
        raise RuntimeError(f"Unexpected API response: {data}")

    return data["result"]
