import logging
import os
from pathlib import Path
from typing import Optional

_log_directory: Optional[Path] = None


class LazyFileHandler(logging.FileHandler):
    """
    A FileHandler that defers file creation until the first log record is emitted.
    This prevents creating log files in the current directory if no logging occurs.
    """
    def __init__(self, filename, mode='a', encoding=None, delay=True):
        """
        Initialize the handler with delay=True to defer file opening.
        """
        # Always use delay=True to defer file creation
        super().__init__(filename, mode, encoding, delay=True)

    def emit(self, record):
        """
        Emit a record, opening the file if needed.
        """
        # The parent class will open the file on first emit if delay=True
        super().emit(record)

def configure_logging(path: str) -> None:
    """
    Configure the logging directory for all csst_fs loggers.

    This function sets the absolute directory where log files will be written.
    All loggers created after calling this function will write their file outputs
    to files within this directory.

    Args:
        path: Absolute path to the logging directory. The directory will be
              created if it doesn't exist.

    Raises:
        ValueError: If the path is not absolute.
        OSError: If the directory cannot be created.

    Example:
        >>> import csst_fs
        >>> csst_fs.configure_logging('/var/log/csst_fs')
    """
    global _log_directory

    path_obj = Path(path)

    if not path_obj.is_absolute():
        raise ValueError(f"Logging path must be absolute, got: {path}")

    # Create directory if it doesn't exist
    path_obj.mkdir(parents=True, exist_ok=True)

    _log_directory = path_obj

    # Update all existing handlers
    _update_existing_handlers()

def get_log_directory() -> Optional[Path]:
    """
    Get the currently configured logging directory.

    Returns:
        Path object of the logging directory, or None if not configured.
    """
    return _log_directory

def get_log_file_path(filename: str) -> str:
    """
    Get the full path for a log file.

    Args:
        filename: Name of the log file.

    Returns:
        Absolute path to the log file. If a logging directory is configured,
        returns path within that directory. Otherwise, returns the filename
        (which will create the log in the current working directory).
    """
    if _log_directory is not None:
        return str(_log_directory / filename)
    return filename

def _update_existing_handlers() -> None:
    """
    Update all existing FileHandlers in csst_fs loggers to use the new directory.
    Only affects loggers within the csst_fs package.
    """
    if _log_directory is None:
        return

    # Iterate through all loggers, but only update csst_fs loggers
    for logger_name in logging.Logger.manager.loggerDict:
        # Only process loggers that belong to the csst_fs package
        if not logger_name.startswith('csst_fs'):
            continue

        logger = logging.getLogger(logger_name)
        if not isinstance(logger, logging.Logger):
            continue

        for handler in logger.handlers[:]:
            if isinstance(handler, logging.FileHandler):
                # Get the base filename
                old_path = Path(handler.baseFilename)
                new_path = _log_directory / old_path.name

                # Create a new lazy handler with the same settings
                new_handler = LazyFileHandler(str(new_path))
                new_handler.setLevel(handler.level)
                new_handler.setFormatter(handler.formatter)

                # Replace the old handler
                logger.removeHandler(handler)
                handler.close()
                logger.addHandler(new_handler)
