import os
import requests
from typing import Dict, Any, List
from pydantic import BaseModel, Field, validator
from ..s3_config import load_backend_settings

class StarCatalogQuery(BaseModel):
    catalogName: str
    key: List[str]
    filter: Dict[str, Any]

    @validator("key")
    def key_not_empty(cls, v):
        if not v:
            raise ValueError("key list cannot be empty")
        return v

class MetadataQuery(BaseModel):
    filter: Dict[str, Any]
    hdu: int = Field(default=0)


class CombinedCatalogMetadataRequest(BaseModel):
    starCatalog: StarCatalogQuery
    metadata: MetadataQuery

def query_star_catalog_with_metadata(
    metadata: Dict[str, Any],
    star_catalog: Dict[str, Any],
) -> List[Dict[str, Any]]:
    """
    Queries the metadata catalog according to the provided filter criteria and HDU value.
    Subsequently queries the star catalog entries corresponding to the metadata results and
    the given additional filters.
    Returns the catalog columns specified in the 'key' list.

    Args:
        metadata: {
            filter: filter dict described in the README filter syntax section.
            hdu: The hdu the filter & key arguments refer to. Default is 0. E.g. 0, 1.
        },
        star_catalog: {
            catalog_name: Name of the star catalog (e.g. csst-msc-l1-mbi-catmix)
            filter: filter dict described in the README filter syntax section.
                The following keys MUST be set:
                {
                    "ra": 40.3,
                    "dec": 21.9,
                    "radius": 0.2,
                }
                Setting ranges with (lt, gt, lte, gte) for ra, dec values is not supported.
            key: A list of string values, corresponding to the column names that should be present in the return value.
        }

        Example:
        from csst_fs import *
        query_starcatalog_with_metadata(
    ...     star_catalog={
    ...         "catalogName": "csst-msc-l1-mbi-catmix",
    ...         "key": ["data_uuid", "obsid", "ra"],
    ...         "filter": {"ra": 130.97, "dec": -20.5, "radius": 0.09, "x": {"lt": 30}},
    ...     },
    ...     metadata={
    ...         "filter": {"priority": {"gte": 2}, "obs_id": 66},
    ...         "hdu": 0,
    ...     }
    ... )

    Returns:
        A List[Dict] of matching star catalog objects.
    """

    api_url = load_backend_settings()['backend_url']
    if not api_url:
        raise RuntimeError("CSST backend api url is not set")

    endpoint = f"{api_url}/starcatalog/query/metadata"

    star_catalog_model = StarCatalogQuery(**star_catalog)
    metadata_model = MetadataQuery(**metadata)

    payload = CombinedCatalogMetadataRequest(
        starCatalog=star_catalog_model,
        metadata=metadata_model,
    ).dict()

    response = requests.post(endpoint, json=payload, timeout=30)
    response.raise_for_status()
    data = response.json()

    if not data.get("success") or "result" not in data:
        raise RuntimeError(f"Unexpected API response: {data}")

    return data["result"]


def query_star_catalog(
    catalog_name: str,
    filter: Dict[str, Any],
    key: List[str],
) -> List[Dict[str, Any]]:
    """
    Query a star catalog by column values given a ra, dec and radius preselection.

    Args:
        catalog_name: Name of the star catalog (e.g. csst-msc-l1-mbi-catmix)
        filter: The filter dict described below.
            The following keys MUST be set:
            {
                "ra": 40.3,
                "dec": 21.9,
                "radius": 0.2,
            }
            Setting ranges with (lt, gt, lte, gte) for ra, dec values is not supported.
        key: A list of string values, corresponding to the column names that should be present in the return value.

    Returns:
        A List[Dict] of matching star catalog objects.
    """

    for required_key in ("ra", "dec", "radius"):
        if required_key not in filter or isinstance(filter[required_key], dict):
            raise ValueError(f"Filter must contain scalar '{{{required_key}}}'")

    if not key:
        raise ValueError("Key list cannot be empty")

    api_url = load_backend_settings()['backend_url']
    if not api_url:
        raise RuntimeError("CSST backend api url is not set")

    endpoint = f"{api_url}/starcatalog/query"

    payload = {
        "catalogName": catalog_name,
        "filter": filter,
        "key": key,
    }

    try:
        response = requests.post(endpoint, json=payload, timeout=30)
        response.raise_for_status()
    except requests.RequestException as e:
        raise RuntimeError(f"Star catalog query failed: {e}")

    data = response.json()

    if not data.get("success") or "result" not in data:
        raise RuntimeError(f"Unexpected API response: {data}")

    return data["result"]
