"""
Identifier:     csst_proto/flip_image.py
Name:           flip_image.py
Description:    这个模块包括了一个图像翻转函数和一个读取随包默认图像的函数
Author:         Bo Zhang
Created:        2023-10-26
Modified-History:
    2023-10-26, Bo Zhang, add module header
    2023-10-28, Bo Zhang, add comments
"""
import multiprocessing
import os
import joblib
import numpy as np
import pathlib

# find the absolute path of this file
# use pathlib
# HERE: pathlib.Path = pathlib.Path(__file__).parent.resolve()
# or use os
HERE: str = os.path.dirname(__file__)


# 此函数翻转了输入图像
# 当use_numpy=True时用numpy函数进行翻转
# 当use_numpy=False时用用指标进行翻转
def flip_image(image: np.ndarray, use_numpy: bool = False) -> np.ndarray:
    """
    Flip an input image.

    This function uses an awesome algorithm to flip an input image.

    Parameters
    ----------
    image : np.ndarray
        The input image.
    use_numpy : bool
        Use numpy.flipud and numpy.fliplr if True, otherwise use indices.

    Returns
    -------
    np.ndarray
        The flipped image.

    Examples
    --------
    >>> flip_image(np.array([[1, 2], [3, 4]]), use_numpy=False)
    array([[4, 3],
           [2, 1]])
    """
    try:
        assert image.ndim == 2

        if use_numpy:
            image_flipped = np.fliplr(np.flipud(image))
        else:
            image_flipped = image[::-1, ::-1]
        return image_flipped
    except AssertionError:
        raise AssertionError("The input image is {}D not 2D!".format(image.ndim))


# 此函数读取了随包数据中的test_image.txt
def read_default_image() -> np.ndarray:
    """
    Read default image.

    This function reads the default image associated with this package.

    Returns
    -------
    np.ndarray
        The test image.

    Examples
    --------
    >>> test_image = read_default_image()
    """
    fp_img = os.path.join(HERE, "data/test_image.txt")
    print("reading file {} ...".format(fp_img))
    return np.loadtxt(fp_img, dtype=int)
