"""
Identifier:     csst_proto/flip_image_in_parallel.py
Name:           flip_image_in_parallel.py
Description:    这个模块包括了基于joblib和multiprocessing的两种并行翻转图像的函数
Author:         Bo Zhang
Created:        2023-10-26
Modified-History:
    2023-10-26, Bo Zhang, add module header
    2023-10-28, Bo Zhang, add comments, separated from flip_image.py
"""
import multiprocessing
import joblib
import numpy as np

from .flip import flip_image


def flip_multiple_images_mp(images: list[np.ndarray], n_jobs: int) -> list[np.ndarray]:
    """
    Flip input images in parallel.

    Flip images in parallel with `multiprocessing`.

    Parameters
    ----------
    images : list[np.ndarray]
        The input image list.
    n_jobs : int
        The number of processes.

    Returns
    -------
    list[np.ndarray]
        The flipped image list.
    """
    with multiprocessing.Pool(n_jobs) as p:
        results = p.map(flip_image, images)
    return results


def flip_multiple_images_jl(images: list[np.ndarray], n_jobs: int) -> list[np.ndarray]:
    """
    Flip input images in parallel.

    Flip images in parallel with `joblib`.

    Parameters
    ----------
    images : list[np.ndarray]
        The input image list.
    n_jobs : int
        The number of processes.

    Returns
    -------
    list[np.ndarray]
        The flipped image list.
    """
    return joblib.Parallel(n_jobs=n_jobs, backend="multiprocessing")(
        joblib.delayed(flip_image)(image) for image in images
    )
